<?php
/**
 * KMG Store Admin Dashboard - Fixed Version with Products & Revenue
 * Main dashboard for administrators with debugging capabilities
 */

session_start();

// Security Headers
header('X-Content-Type-Options: nosniff');
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Referrer-Policy: strict-origin-when-cross-origin');

// Authentication Check
if (!isset($_SESSION['admin_logged_in']) || $_SESSION['admin_logged_in'] !== true) {
    header("Location: admin.html");
    exit();
}

// IP Validation
if (!isset($_SESSION['admin_ip'])) {
    $_SESSION['admin_ip'] = $_SERVER['REMOTE_ADDR'];
} elseif ($_SESSION['admin_ip'] !== $_SERVER['REMOTE_ADDR']) {
    session_destroy();
    header("Location: admin.html");
    exit();
}

// Session Timeout (30 minutes)
$timeout_duration = 1800;
if (isset($_SESSION['last_activity']) && (time() - $_SESSION['last_activity']) > $timeout_duration) {
    session_destroy();
    header("Location: admin.html");
    exit();
}
$_SESSION['last_activity'] = time();

// CSRF Token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

/**
 * Database Manager - Singleton Pattern
 */
class DatabaseManager {
    private static $instance = null;
    private $connection;
    
    // Database Configuration
    private $host = "localhost";
    private $username = "root";
    private $password = "";
    private $database = "kmgstore";
    
    private function __construct() {
        $this->connect();
    }
    
    private function connect() {
        try {
            $this->connection = new mysqli($this->host, $this->username, $this->password, $this->database);
            
            if ($this->connection->connect_error) {
                throw new Exception("Database connection failed: " . $this->connection->connect_error);
            }
            
            // Set charset and timezone
            $this->connection->set_charset("utf8mb4");
            $this->connection->query("SET sql_mode = 'STRICT_TRANS_TABLES,ERROR_FOR_DIVISION_BY_ZERO,NO_AUTO_CREATE_USER,NO_ENGINE_SUBSTITUTION'");
            $this->connection->query("SET time_zone = '+07:00'");
            
        } catch (Exception $e) {
            error_log("Database connection error: " . $e->getMessage());
            http_response_code(500);
            die("Database connection failed. Please try again later.");
        }
    }
    
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new DatabaseManager();
        }
        return self::$instance;
    }
    
    public function getConnection() {
        // Check if connection is alive
        if (!$this->connection->ping()) {
            $this->connect();
        }
        return $this->connection;
    }
    
    public function close() {
        if ($this->connection) {
            $this->connection->close();
        }
    }
    
    public function prepare($query) {
        $stmt = $this->connection->prepare($query);
        if (!$stmt) {
            error_log("Prepare failed: " . $this->connection->error);
            throw new Exception("Database prepare error");
        }
        return $stmt;
    }
    
    public function safeQuery($query, $params = [], $types = "") {
        try {
            if (empty($params)) {
                $result = $this->connection->query($query);
                if (!$result) {
                    throw new Exception("Query failed: " . $this->connection->error);
                }
                return $result;
            }
            
            $stmt = $this->prepare($query);
            if (!empty($types) && !empty($params)) {
                $stmt->bind_param($types, ...$params);
            }
            $stmt->execute();
            return $stmt->get_result();
            
        } catch (Exception $e) {
            error_log("Database query error: " . $e->getMessage());
            return false;
        }
    }
    
    public function getTableColumns($tableName) {
        $query = "SHOW COLUMNS FROM " . $this->connection->real_escape_string($tableName);
        $result = $this->safeQuery($query);
        
        $columns = [];
        if ($result) {
            while ($column = $result->fetch_assoc()) {
                $columns[] = $column['Field'];
            }
        }
        return $columns;
    }
    
    public function tableExists($tableName) {
        $query = "SHOW TABLES LIKE '" . $this->connection->real_escape_string($tableName) . "'";
        $result = $this->safeQuery($query);
        return $result && $result->num_rows > 0;
    }
}

/**
 * Statistics Cache - Simple caching system
 */
class StatsCache {
    private static $cache = [];
    private static $cacheTTL = 300; // 5 minutes
    
    public static function get($key) {
        if (isset(self::$cache[$key])) {
            $data = self::$cache[$key];
            if (time() - $data['timestamp'] < self::$cacheTTL) {
                return $data['value'];
            }
            unset(self::$cache[$key]);
        }
        return null;
    }
    
    public static function set($key, $value) {
        self::$cache[$key] = [
            'value' => $value,
            'timestamp' => time()
        ];
    }
}

/**
 * Utility Functions
 */
function sanitizeInput($input, $maxLength = 255) {
    if ($input === null) return 'N/A';
    $input = trim($input);
    $input = substr($input, 0, $maxLength);
    return htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
}

function formatCurrency($amount) {
    return number_format($amount, 0, ',', '.');
}

function formatDate($date, $format = 'd/m/Y H:i') {
    if (!$date || $date === '0000-00-00 00:00:00') return 'N/A';

    if (is_numeric($date)) {
        return date($format, $date);
    }
    
    $timestamp = strtotime($date);
    if ($timestamp === false) {
        $formats = ['Y-m-d H:i:s', 'Y-m-d', 'd/m/Y H:i:s', 'd/m/Y'];
        foreach ($formats as $f) {
            $dateObj = DateTime::createFromFormat($f, $date);
            if ($dateObj !== false) {
                return $dateObj->format($format);
            }
        }
        return 'N/A';
    }
    
    return date($format, $timestamp);
}

/**
 * Get Dashboard Statistics - FIXED VERSION
 */
function getStatistics($db) {
    $cacheKey = 'dashboard_stats';
    $cached = StatsCache::get($cacheKey);
    
    if ($cached !== null) {
        return $cached;
    }
    
    $stats = [
        'totalTransaksi' => 0,
        'totalPengguna' => 0,
        'totalProduk' => 0,
        'totalNominal' => 0,
        'todayTransactions' => 0,
        'todayRevenue' => 0,
        'monthlyGrowth' => 0
    ];
    
    try {
        // Total Transactions
        $result = $db->safeQuery("SELECT COUNT(*) AS total FROM transaksi");
        if ($result) {
            $stats['totalTransaksi'] = $result->fetch_assoc()['total'] ?? 0;
        }
        
        // Total Users - Check multiple possible user table names
        $userTables = ['users', 'user', 'customer', 'pengguna'];
        foreach ($userTables as $table) {
            if ($db->tableExists($table)) {
                $result = $db->safeQuery("SELECT COUNT(*) AS total FROM $table");
                if ($result) {
                    $stats['totalPengguna'] = $result->fetch_assoc()['total'] ?? 0;
                }
                break;
            }
        }
        
        // Total Products - Check multiple possible product table names
        $productTables = ['products', 'product', 'produk', 'items', 'game'];
        foreach ($productTables as $table) {
            if ($db->tableExists($table)) {
                $result = $db->safeQuery("SELECT COUNT(*) AS total FROM $table");
                if ($result) {
                    $stats['totalProduk'] = $result->fetch_assoc()['total'] ?? 0;
                }
                break;
            }
        }
        
        // FIXED: Total Revenue with better column detection
        $transaksiColumns = $db->getTableColumns('transaksi');
        $revenueColumns = ['total_bayar', 'total_harga', 'nominal', 'harga', 'amount', 'price'];
        $foundRevenueColumn = null;
        
        foreach ($revenueColumns as $col) {
            if (in_array($col, $transaksiColumns)) {
                $foundRevenueColumn = $col;
                break;
            }
        }
        
        if ($foundRevenueColumn) {
            $revenueQuery = "
                SELECT COALESCE(SUM(CAST($foundRevenueColumn AS DECIMAL(15,2))), 0) AS total_nominal 
                FROM transaksi 
                WHERE status IN ('berhasil', 'Success', 'Completed', 'success', 'selesai', 'paid', 'complete')
                AND $foundRevenueColumn IS NOT NULL 
                AND $foundRevenueColumn > 0
            ";
            $result = $db->safeQuery($revenueQuery);
            if ($result) {
                $stats['totalNominal'] = $result->fetch_assoc()['total_nominal'] ?? 0;
            }
        }
        
        // Today's Statistics - Check for date columns
        $dateColumns = ['created_at', 'tanggal', 'date_created', 'timestamp', 'waktu', 'tgl_transaksi'];
        $dateColumn = null;
        
        foreach ($dateColumns as $col) {
            if (in_array($col, $transaksiColumns)) {
                $dateColumn = $col;
                break;
            }
        }
        
        if ($dateColumn && $foundRevenueColumn) {
            $todayQuery = "
                SELECT 
                    COUNT(*) as transactions_today,
                    COALESCE(SUM(CAST($foundRevenueColumn AS DECIMAL(15,2))), 0) as revenue_today
                FROM transaksi 
                WHERE DATE($dateColumn) = CURDATE()
                AND status IN ('berhasil', 'Success', 'Completed', 'success', 'selesai', 'paid', 'complete')
                AND $foundRevenueColumn IS NOT NULL 
                AND $foundRevenueColumn > 0
            ";
            $result = $db->safeQuery($todayQuery);
            if ($result) {
                $todayData = $result->fetch_assoc();
                $stats['todayTransactions'] = $todayData['transactions_today'] ?? 0;
                $stats['todayRevenue'] = $todayData['revenue_today'] ?? 0;
            }
            
            // Monthly Growth
            $growthQuery = "
                SELECT 
                    COUNT(CASE WHEN MONTH($dateColumn) = MONTH(CURDATE()) AND YEAR($dateColumn) = YEAR(CURDATE()) THEN 1 END) as this_month,
                    COUNT(CASE WHEN MONTH($dateColumn) = MONTH(CURDATE()) - 1 AND YEAR($dateColumn) = YEAR(CURDATE()) THEN 1 END) as last_month
                FROM transaksi 
                WHERE $dateColumn IS NOT NULL
            ";
            $result = $db->safeQuery($growthQuery);
            if ($result) {
                $growthData = $result->fetch_assoc();
                $thisMonth = $growthData['this_month'] ?? 0;
                $lastMonth = $growthData['last_month'] ?? 0;
                
                if ($lastMonth > 0) {
                    $stats['monthlyGrowth'] = round((($thisMonth - $lastMonth) / $lastMonth) * 100, 1);
                }
            }
        }
        
    } catch (Exception $e) {
        error_log("Statistics error: " . $e->getMessage());
    }
    
    StatsCache::set($cacheKey, $stats);
    return $stats;
}

/**
 * Get Recent Transactions - Fixed Version
 */
function getRecentTransactions($db) {
    try {
        // Get table columns
        $columns = $db->getTableColumns('transaksi');
        
        // Find the best date column for ordering
        $dateColumns = ['created_at', 'tanggal', 'date_created', 'timestamp', 'waktu', 'tgl_transaksi'];
        $dateColumn = 'id'; // fallback
        
        foreach ($dateColumns as $col) {
            if (in_array($col, $columns)) {
                $dateColumn = $col;
                break;
            }
        }
        
        // Find revenue column
        $revenueColumns = ['total_bayar', 'total_harga', 'nominal', 'harga', 'amount', 'price'];
        $revenueColumn = null;
        
        foreach ($revenueColumns as $col) {
            if (in_array($col, $columns)) {
                $revenueColumn = $col;
                break;
            }
        }
        
        // Check for user tables
        $userTables = ['user', 'users', 'customer', 'pengguna'];
        $userTable = null;
        $userIdColumn = 'id';
        $userNameColumn = 'name';
        
        foreach ($userTables as $table) {
            if ($db->tableExists($table)) {
                $userTable = $table;
                $userColumns = $db->getTableColumns($table);
                
                // Find user ID column
                $userIdOptions = ['id', 'user_id', 'User_id', 'ID'];
                foreach ($userIdOptions as $idCol) {
                    if (in_array($idCol, $userColumns)) {
                        $userIdColumn = $idCol;
                        break;
                    }
                }
                
                // Find user name column
                $userNameOptions = ['name', 'Name', 'username', 'user_name', 'nama', 'full_name'];
                foreach ($userNameOptions as $nameCol) {
                    if (in_array($nameCol, $userColumns)) {
                        $userNameColumn = $nameCol;
                        break;
                    }
                }
                break;
            }
        }
        
        // Build query
        $revenueSelect = $revenueColumn ? "t.$revenueColumn" : "0";
        
        if ($userTable) {
            $query = "
                SELECT 
                    t.id, 
                    t.kode_transaksi, 
                    t.status, 
                    $revenueSelect as nominal,
                    " . (in_array('nama_game', $columns) ? "t.nama_game," : "'' as nama_game,") . "
                    t.$dateColumn as created_at, 
                    t.user_id,
                    COALESCE(u.$userNameColumn, 'N/A') as user_name
                FROM transaksi t
                LEFT JOIN $userTable u ON t.user_id = u.$userIdColumn
                ORDER BY t.$dateColumn DESC, t.id DESC
                LIMIT 10
            ";
        } else {
            $query = "
                SELECT 
                    t.id, 
                    t.kode_transaksi, 
                    t.status, 
                    $revenueSelect as nominal,
                    " . (in_array('nama_game', $columns) ? "t.nama_game," : "'' as nama_game,") . "
                    t.$dateColumn as created_at, 
                    t.user_id,
                    'N/A' as user_name
                FROM transaksi t
                ORDER BY t.$dateColumn DESC, t.id DESC
                LIMIT 10
            ";
        }
        
        $result = $db->safeQuery($query);
        
        // If no results, try simpler query
        if (!$result || $result->num_rows == 0) {
            $simpleQuery = "
                SELECT 
                    id, 
                    kode_transaksi, 
                    status, 
                    " . ($revenueColumn ? $revenueColumn : "0") . " as nominal,
                    " . (in_array('nama_game', $columns) ? "nama_game," : "'' as nama_game,") . "
                    user_id,
                    $dateColumn as created_at,
                    'N/A' as user_name
                FROM transaksi 
                ORDER BY id DESC 
                LIMIT 10
            ";
            $result = $db->safeQuery($simpleQuery);
        }
        
        return $result;
        
    } catch (Exception $e) {
        error_log("Recent transactions error: " . $e->getMessage());
        return false;
    }
}

/**
 * Get Game Statistics
 */
function getGameStatistics($db) {
    $columns = $db->getTableColumns('transaksi');
    
    if (!in_array('nama_game', $columns)) {
        return false;
    }
    
    // Find revenue column
    $revenueColumns = ['total_bayar', 'total_harga', 'nominal', 'harga', 'amount', 'price'];
    $revenueColumn = 'nominal'; // default
    
    foreach ($revenueColumns as $col) {
        if (in_array($col, $columns)) {
            $revenueColumn = $col;
            break;
        }
    }
    
    $query = "
        SELECT 
            nama_game,
            COUNT(*) as total_transaksi,
            COUNT(CASE WHEN status IN ('berhasil', 'Success', 'success', 'Completed', 'selesai', 'paid', 'complete') THEN 1 END) as successful_transactions,
            COALESCE(SUM(
                CASE 
                    WHEN status IN ('berhasil', 'Success', 'success', 'Completed', 'selesai', 'paid', 'complete') 
                    AND $revenueColumn IS NOT NULL 
                    AND $revenueColumn > 0 
                    THEN CAST($revenueColumn AS DECIMAL(15,2))
                    ELSE 0
                END
            ), 0) as total_pendapatan
        FROM transaksi 
        WHERE nama_game IS NOT NULL AND nama_game != '' AND nama_game != 'NULL'
        GROUP BY nama_game
        ORDER BY total_pendapatan DESC
        LIMIT 5
    ";
    
    return $db->safeQuery($query);
}

/**
 * Get Transaction Statistics by Status
 */
function getTransactionStatsByStatus($db) {
    $stats = [];
    
    // Success transactions
    $successQuery = "SELECT COUNT(*) as total FROM transaksi WHERE status IN ('berhasil', 'Success', 'success', 'Completed', 'selesai', 'paid', 'complete')";
    $result = $db->safeQuery($successQuery);
    $stats['success'] = $result ? $result->fetch_assoc()['total'] : 0;
    
    // Pending transactions
    $pendingQuery = "SELECT COUNT(*) as total FROM transaksi WHERE status IN ('pending', 'Pending', 'waiting', 'menunggu', 'process', 'processing')";
    $result = $db->safeQuery($pendingQuery);
    $stats['pending'] = $result ? $result->fetch_assoc()['total'] : 0;
    
    // Failed transactions
    $failedQuery = "SELECT COUNT(*) as total FROM transaksi WHERE status IN ('failed', 'Failed', 'cancelled', 'Cancelled', 'gagal', 'batal', 'cancel', 'reject')";
    $result = $db->safeQuery($failedQuery);
    $stats['failed'] = $result ? $result->fetch_assoc()['total'] : 0;
    
    return $stats;
}

// Initialize Database
$db = DatabaseManager::getInstance();

// Debug mode - uncomment to see table structure
$debug = isset($_GET['debug']) && $_GET['debug'] === 'true';
if ($debug) {
    echo "<div style='background: #f8f9fa; padding: 20px; margin: 20px; border-radius: 8px; font-family: monospace;'>";
    echo "<h3>Debug Information</h3>";
    echo "<strong>Transaksi table columns:</strong> " . implode(', ', $db->getTableColumns('transaksi')) . "<br>";
    echo "<strong>User table exists:</strong> " . ($db->tableExists('user') ? 'Yes' : 'No') . "<br>";
    echo "<strong>Users table exists:</strong> " . ($db->tableExists('users') ? 'Yes' : 'No') . "<br>";
    echo "<strong>Products table exists:</strong> " . ($db->tableExists('products') ? 'Yes' : 'No') . "<br>";
    echo "<strong>Product table exists:</strong> " . ($db->tableExists('product') ? 'Yes' : 'No') . "<br>";
    echo "<strong>Produk table exists:</strong> " . ($db->tableExists('produk') ? 'Yes' : 'No') . "<br>";
    echo "</div>";
}

// Get Dashboard Data
$stats = getStatistics($db);
$recentTransactions = getRecentTransactions($db);
$gameStatsResult = getGameStatistics($db);
$transactionStats = getTransactionStatsByStatus($db);

// Calculate average transaction amount
$avgAmount = $stats['totalTransaksi'] > 0 ? $stats['totalNominal'] / $stats['totalTransaksi'] : 0;

// Cleanup on shutdown
register_shutdown_function(function() use ($db) {
    $db->close();
});
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>KMG Store - Admin Dashboard</title>
    
    <!-- CSS Dependencies -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.0/css/all.min.css" rel="stylesheet">
    
    <style>
        :root {
            --primary-color: #2563eb;
            --primary-dark: #1d4ed8;
            --success-color: #16a34a;
            --success-dark: #15803d;
            --warning-color: #f59e0b;
            --warning-dark: #d97706;
            --danger-color: #dc2626;
            --danger-dark: #b91c1c;
            --bg-color: #f8fafc;
            --border-color: #e5e7eb;
            --text-color: #374151;
            --text-muted: #6b7280;
            --sidebar-width: 250px;
            --border-radius: 12px;
            --box-shadow: 0 2px 10px rgba(0,0,0,0.05);
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8fafc;
            margin: 0;
        }

       /* Sidebar Styles */
        .sidebar {
            width: var(--sidebar-width);
            position: fixed;
            height: 100vh;
            background-color: #ffffff;
            border-right: 1px solid var(--border-color);
            padding: 2rem 1.5rem;
            box-shadow: var(--box-shadow);
            z-index: 1000;
        }

        .sidebar .logo {
            margin-bottom: 2rem;
            text-align: center;
            border-bottom: 1px solid var(--border-color);
            padding-bottom: 1rem;
        }

        .sidebar .logo h4 {
            color: var(--primary-color);
            font-weight: 700;
            margin-bottom: 0.5rem;
        }

        .sidebar .logo small {
            color: var(--text-muted);
            font-size: 0.875rem;
        }

        .sidebar .nav-link {
            color: var(--text-color);
            padding: 0.875rem 1rem;
            margin-bottom: 0.5rem;
            border-radius: 8px;
            text-decoration: none;
            display: flex;
            align-items: center;
            transition: all 0.3s ease;
            font-weight: 500;
        }

        .sidebar .nav-link:hover {
            background-color: #f0f9ff;
            color: var(--primary-color);
            transform: translateX(4px);
        }

        .sidebar .nav-link.active {
            background-color: var(--primary-color);
            color: white;
            box-shadow: 0 4px 12px rgba(37, 99, 235, 0.3);
        }

        .sidebar .nav-link i {
            margin-right: 0.75rem;
            width: 18px;
            text-align: center;
        }

        .sidebar hr {
            border: 0;
            border-top: 1px solid var(--border-color);
            margin: 1.5rem 0;
        }

        .sidebar .nav-link.text-danger:hover {
            background-color: #fef2f2;
            color: var(--danger-color);
        }

        /* Main Content */
        .main-content {
            margin-left: var(--sidebar-width);
            padding: 2rem;
            min-height: 100vh;
        }

        /* Cards */
        .card-custom {
            border-radius: 16px;
            box-shadow: var(--card-shadow);
            border: none;
        }

        .card-hover {
            transition: all 0.3s ease;
        }

        .card-hover:hover {
            transform: translateY(-2px);
            box-shadow: var(--hover-shadow);
        }

        /* Status Badges */
        .status-badge {
            padding: 0.25rem 0.5rem;
            border-radius: 0.375rem;
            font-size: 0.75rem;
            font-weight: 500;
        }

        .status-pending { 
            background-color: #fef3c7; 
            color: #92400e; 
        }

        .status-berhasil, 
        .status-success { 
            background-color: #d1fae5; 
            color: #065f46; 
        }

        .status-failed { 
            background-color: #fee2e2; 
            color: #991b1b; 
        }

        .status-cancelled { 
            background-color: #f3f4f6; 
            color: #374151; 
        }

        /* Game List */
        .game-list-item {
            padding: 12px 0;
            border-bottom: 1px solid #f3f4f6;
            display: flex;
            align-items: center;
            justify-content: space-between;
        }

        .game-list-item:last-child {
            border-bottom: none;
        }

        .game-list-item:first-child {
            padding-top: 0;
        }

        .game-info {
            flex: 1;
            min-width: 0;
        }

        .game-price {
            flex-shrink: 0;
            margin-left: 16px;
        }

        /* Table Styles */
        .table th {
            background-color: #f1f5f9;
            border-bottom: 2px solid #e2e8f0;
            font-weight: 600;
        }

        /* Responsive Design */
        @media (max-width: 768px) {
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
            }

            .sidebar.open {
                transform: translateX(0);
            }

            .main-content {
                margin-left: 0;
                padding: 20px;
            }
        }

        /* Loading Animation */
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-radius: 50%;
            border-top: 3px solid var(--primary-color);
            animation: spin 1s linear infinite;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        /* Custom Scrollbar */
        ::-webkit-scrollbar {
            width: 8px;
        }

        ::-webkit-scrollbar-track {
            background: #f1f1f1;
        }

        ::-webkit-scrollbar-thumb {
            background: #c1c1c1;
            border-radius: 4px;
        }

        ::-webkit-scrollbar-thumb:hover {
            background: #a1a1a1;
        }

        /* Responsive Design */
        @media (max-width: 768px) 
            .sidebar {
                transform: translateX(-100%);
                transition: transform 0.3s ease;
            }
    </style>
</head>
<body>
     <!-- Sidebar -->
    <div class="sidebar">
        <div class="logo">
            <h4><i class="fas fa-store"></i> KMG Store</h4>
            <small>Admin Panel</small>
        </div>

        <nav class="nav flex-column">
            <a class="nav-link active" href="admin-dashboard.php">
                <i class="fas fa-tachometer-alt"></i> Dashboard
            </a>
            <a class="nav-link" href="transaksi.php">
                <i class="fas fa-shopping-cart"></i> Transaksi
            </a>
            <a class="nav-link" href="pengguna.php">
                <i class="fas fa-users"></i> Pengguna
            </a>
            <a class="nav-link" href="productlist.php">
                <i class="fas fa-boxes"></i> Product List
            </a>
            <a class="nav-link" href="add_product.php">
                <i class="fas fa-plus-circle"></i> Tambah Produk
            </a>
            
            <hr>
            
            <a class="nav-link text-danger" href="logout.php">
                <i class="fas fa-sign-out-alt"></i> Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div class="d-flex align-items-center">
                <button class="btn btn-outline-secondary d-md-none me-3" id="mobile-menu-btn">
                    <i class="fas fa-bars"></i>
                </button>
                <h1 class="h3 mb-0">Dashboard</h1>
            </div>
            
            <div class="d-flex align-items-center">
                <button onclick="location.reload()" class="btn btn-primary me-3">
                    <i class="fas fa-sync-alt me-2"></i>
                    Refresh
                </button>
                
                <div class="d-none<div class="dropdown">
                    <button class="btn btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                        <i class="fas fa-user me-2"></i>
                        <?php echo sanitizeInput($_SESSION['admin_name'] ?? 'Admin'); ?>
                    </button>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row g-4 mb-4">
            <div class="col-xl-3 col-md-6">
                <div class="card card-custom card-hover">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="flex-shrink-0">
                                <div class="bg-primary bg-opacity-10 rounded-circle p-3">
                                    <i class="fas fa-shopping-cart text-primary fa-lg"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h3 class="mb-1"><?php echo number_format($stats['totalTransaksi']); ?></h3>
                                <p class="text-muted mb-0 small">Total Transaksi</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-xl-3 col-md-6">
                <div class="card card-custom card-hover">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="flex-shrink-0">
                                <div class="bg-success bg-opacity-10 rounded-circle p-3">
                                    <i class="fas fa-users text-success fa-lg"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h3 class="mb-1"><?php echo number_format($stats['totalPengguna']); ?></h3>
                                <p class="text-muted mb-0 small">Total Pengguna</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-xl-3 col-md-6">
                <div class="card card-custom card-hover">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="flex-shrink-0">
                                <div class="bg-warning bg-opacity-10 rounded-circle p-3">
                                    <i class="fas fa-box text-warning fa-lg"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h3 class="mb-1"><?php echo number_format($stats['totalProduk']); ?></h3>
                                <p class="text-muted mb-0 small">Total Produk</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-xl-3 col-md-6">
                <div class="card card-custom card-hover">
                    <div class="card-body">
                        <div class="d-flex align-items-center">
                            <div class="flex-shrink-0">
                                <div class="bg-info bg-opacity-10 rounded-circle p-3">
                                    <i class="fas fa-money-bill-wave text-info fa-lg"></i>
                                </div>
                            </div>
                            <div class="flex-grow-1 ms-3">
                                <h3 class="mb-1">Rp <?php echo formatCurrency($stats['totalNominal']); ?></h3>
                                <p class="text-muted mb-0 small">Total Pendapatan</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        

        <!-- Recent Transactions and Popular Games -->
        <div class="row g-4 mb-4">
            <div class="col-lg-8">
                <div class="card card-custom">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">Transaksi Terbaru</h5>
                        <a href="transaksi.php" class="btn btn-sm btn-primary">
                            Lihat Semua <i class="fas fa-arrow-right ms-1"></i>
                        </a>
                    </div>
                    <div class="card-body">
                        <?php if ($recentTransactions && $recentTransactions->num_rows > 0): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Kode Transaksi</th>
                                            <th>Game</th>
                                            <th>User</th>
                                            <th>Nominal</th>
                                            <th>Status</th>
                                            <th>Tanggal</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($row = $recentTransactions->fetch_assoc()): ?>
                                        <tr>
                                            <td><?php echo sanitizeInput($row['id']); ?></td>
                                            <td>
                                                <code><?php echo sanitizeInput($row['kode_transaksi']); ?></code>
                                            </td>
                                            <td>
                                                <?php 
                                                    $game = sanitizeInput($row['nama_game'] ?? 'N/A');
                                                    if (strlen($game) > 20) {
                                                        $game = substr($game, 0, 20) . '...';
                                                    }
                                                    echo $game;
                                                ?>
                                            </td>
                                            <td>
                                                <?php 
                                                    $userName = sanitizeInput($row['user_name'] ?? 'N/A');
                                                    if (strlen($userName) > 15) {
                                                        $userName = substr($userName, 0, 15) . '...';
                                                    }
                                                    echo $userName;
                                                ?>
                                            </td>
                                            <td>
                                                <strong>Rp <?php echo formatCurrency($row['nominal'] ?? 0); ?></strong>
                                            </td>
                                            <td>
                                                <?php 
                                                    $status = strtolower(sanitizeInput($row['status']));
                                                    $statusClass = 'status-' . $status;
                                                    if (in_array($status, ['berhasil', 'success', 'completed', 'selesai', 'paid', 'complete'])) {
                                                        $statusClass = 'status-berhasil';
                                                    } elseif (in_array($status, ['pending', 'waiting', 'menunggu', 'process', 'processing'])) {
                                                        $statusClass = 'status-pending';
                                                    } elseif (in_array($status, ['failed', 'cancelled', 'gagal', 'batal', 'cancel', 'reject'])) {
                                                        $statusClass = 'status-failed';
                                                    }
                                                ?>
                                                <span class="status-badge <?php echo $statusClass; ?>">
                                                    <?php echo ucfirst($status); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <small><?php echo formatDate($row['created_at']); ?></small>
                                            </td>
                                        </tr>
                                        <?php endwhile; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-4">
                                <i class="fas fa-inbox fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Belum ada transaksi</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <div class="col-lg-4">
                <div class="card card-custom">
                    <div class="card-header">
                        <h5 class="card-title mb-0">Game Populer</h5>
                    </div>
                    <div class="card-body">
                        <?php if ($gameStatsResult && $gameStatsResult->num_rows > 0): ?>
                            <?php $rank = 1; ?>
                            <?php while ($game = $gameStatsResult->fetch_assoc()): ?>
                            <div class="game-list-item">
                                <div class="game-info">
                                    <div class="d-flex align-items-center">
                                        <span class="badge bg-primary me-2"><?php echo $rank; ?></span>
                                        <div>
                                            <div class="fw-bold">
                                                <?php 
                                                    $gameName = sanitizeInput($game['nama_game']);
                                                    if (strlen($gameName) > 25) {
                                                        $gameName = substr($gameName, 0, 25) . '...';
                                                    }
                                                    echo $gameName;
                                                ?>
                                            </div>
                                            <small class="text-muted">
                                                <?php echo number_format($game['successful_transactions']); ?> transaksi
                                            </small>
                                        </div>
                                    </div>
                                </div>
                                <div class="game-price">
                                    <div class="fw-bold text-success">
                                        Rp <?php echo formatCurrency($game['total_pendapatan']); ?>
                                    </div>
                                </div>
                            </div>
                            <?php $rank++; ?>
                            <?php endwhile; ?>
                        <?php else: ?>
                            <div class="text-center py-4">
                                <i class="fas fa-gamepad fa-3x text-muted mb-3"></i>
                                <p class="text-muted">Belum ada data game</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

       

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    
    <script>
        // Mobile menu toggle
        document.getElementById('mobile-menu-btn')?.addEventListener('click', function() {
            const sidebar = document.getElementById('sidebar');
            sidebar.classList.toggle('open');
        });

        // Auto-refresh every 5 minutes
        setInterval(function() {
            const refreshBtn = document.querySelector('[onclick="location.reload()"]');
            if (refreshBtn) {
                refreshBtn.innerHTML = '<i class="fas fa-sync-alt fa-spin me-2"></i>Refreshing...';
                setTimeout(() => {
                    location.reload();
                }, 1000);
            }
        }, 300000);

        // CSRF token for AJAX requests
        const csrfToken = '<?php echo $_SESSION['csrf_token']; ?>';

        // Function to format numbers with animation
        function animateNumber(element, start, end, duration) {
            let startTime = null;
            
            function animate(currentTime) {
                if (startTime === null) startTime = currentTime;
                const timeElapsed = currentTime - startTime;
                const progress = Math.min(timeElapsed / duration, 1);
                
                const currentNumber = Math.floor(progress * (end - start) + start);
                element.textContent = currentNumber.toLocaleString();
                
                if (progress < 1) {
                    requestAnimationFrame(animate);
                }
            }
            
            requestAnimationFrame(animate);
        }

        // Initialize animations when page loads
        document.addEventListener('DOMContentLoaded', function() {
            // Animate statistics cards
            const statsCards = document.querySelectorAll('.card-custom h3');
            statsCards.forEach(card => {
                const text = card.textContent.replace(/[^\d]/g, '');
                const number = parseInt(text) || 0;
                if (number > 0) {
                    card.textContent = '0';
                    setTimeout(() => {
                        animateNumber(card, 0, number, 1000);
                    }, 200);
                }
            });

            // Initialize tooltips
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(function (tooltipTriggerEl) {
                return new bootstrap.Tooltip(tooltipTriggerEl);
            });
        });

        // Handle sidebar navigation
        document.querySelectorAll('.sidebar .nav-link').forEach(link => {
            link.addEventListener('click', function(e) {
                if (this.getAttribute('href') === '#') {
                    e.preventDefault();
                    return;
                }
                
                // Remove active class from all links
                document.querySelectorAll('.sidebar .nav-link').forEach(l => l.classList.remove('active'));
                // Add active class to clicked link
                this.classList.add('active');
            });
        });

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                if (alert.classList.contains('alert-dismissible')) {
                    alert.classList.add('fade');
                    setTimeout(() => alert.remove(), 500);
                }
            });
        }, 5000);

        // Real-time clock
        function updateClock() {
            const now = new Date();
            const timeString = now.toLocaleString('id-ID', {
                weekday: 'long',
                year: 'numeric',
                month: 'long',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            });
            
            const clockElement = document.getElementById('real-time-clock');
            if (clockElement) {
                clockElement.textContent = timeString;
            }
        }

        // Update clock every second
        setInterval(updateClock, 1000);
        updateClock(); // Initial call

        // Debug information toggle
        if (window.location.search.includes('debug=true')) {
            console.log('Debug mode enabled');
            console.log('CSRF Token:', csrfToken);
            console.log('Session Data:', {
                admin_logged_in: true,
                timestamp: new Date().toISOString()
            });
        }
    </script>
</body>
</html>